﻿
using System;
using System.Drawing;
using System.Windows.Forms;

/* Integration of a given function f(x) by the Newton-Cotes formulation */
/* www.mosismath.com                                                    */

namespace WindowsFormsApplication1
{

    public partial class MainWin : Form
    {
        public const int order = 8;         // number of used supportingpoints
        public const int datapoints = 1000; // number of datapoints for the interpolation
      
        double integral = 0;
        double a = 0;
        double b = Math.PI / 2;
        public double deltaX;

        double[] yp = new double[datapoints];
        double[] xp = new double[datapoints];
        double[] y_k = new double[order+1];
        double[] x_k = new double[order+1];
        double[] w = new double[order + 1];

        private double f_x(double x)
        {
            return 5.0 / (Math.Exp(Math.PI) - 2.0) * Math.Exp(2 * x) * Math.Cos(x);
        }

              
        public MainWin()
        {         
            InitializeComponent();
        }

        private double[] MultPoly(double[] p1, double[] p2)
        {
            int i, j;
            double[] tempPoly = new double[p1.Length + p2.Length - 1];
            for (i = 0; i < p1.Length; i++)
            {
                for (j = 0; j < p2.Length; j++)
                {
                    tempPoly[j + i] = tempPoly[j + i] + p2[j] * p1[i];
                }
            }
            return tempPoly;
        }

        private double CalcWeights(int index)
        {
            int i;
            double res = 0;
            double[] wl = new double[1];
            double[] poly = new double[2];
            wl[0] = 1;
            // build polynomial
            for (i = 0; i < x_k.Length; i++)
            {
                if (i != index)
                {
                    poly[0] = -x_k[i] / (x_k[index] - x_k[i]);
                    poly[1] = 1 / (x_k[index] - x_k[i]);
                    wl = MultPoly(wl, poly);
                }
            }
            // build integral
            res = wl[0] * (b - a);
            for (i = 1; i < x_k.Length; i++)
            {
                res = res + wl[i] * (Math.Pow(b, i + 1) - Math.Pow(a, i + 1)) / (i + 1);
            }
            return res;
        }

        public void Interpolate(int maxPoints, double maxTime)
        {
            int i;
            for (i = 0; i < maxPoints; i++)
            {
                yp[i] = 0;
                xp[i] = (double)(i) * maxTime / (double)maxPoints;
                yp[i] = f_x(xp[i]);
            }
        }

        public void DrawGraph(int maxPoints, double minX, double maxX, bool doClear)
        {
            Point p1, p2;
            int j;
            double maxTime = -1.0;
            double maxVal = -1.0;
            double minVal = 1.0;
            double scalefactor;
            bool drawCeroline;
            for (j = 0; j < maxPoints - 1; j++)
            {
                if ((xp[j] >= minX) && (xp[j] <= maxX))
                {
                    if (maxTime < xp[j])
                        maxTime = xp[j];
                    if (maxVal < yp[j])
                        maxVal = yp[j];
                    if (minVal > yp[j])
                        minVal = yp[j];
                }
            }
            maxTime = maxTime * 1.05;
            maxTime = maxTime * 1.05;
            if (minVal < 0)
            {
                if (maxVal > 0)
                {
                    if (maxVal > Math.Abs(minVal))
                        scalefactor = pGraph.Height / maxVal / 2.2;
                    else
                        scalefactor = pGraph.Height / Math.Abs(minVal) / 2.2;
                }
                else
                    scalefactor = 5.0;
                drawCeroline = true;
            }
            else
            {
                scalefactor = pGraph.Height / (maxVal) / 1.1;
                drawCeroline = false;
            }
            p1 = new Point();
            p2 = new Point();
            Graphics g = pGraph.CreateGraphics();
            if (doClear)
                g.Clear(Color.White);
            Pen bluePen = new Pen(Color.Blue, 2);
            Pen redPen = new Pen(Color.Red, 2);
            Pen blackPen = new Pen(Color.Black, 2);
            if (drawCeroline)
            {
                p1.X = 0;
                p1.Y = (pGraph.Height / 2);
                p2.X = pGraph.Width;
                p2.Y = (pGraph.Height / 2);
                g.DrawLine(blackPen, p1, p2);
            }
            bluePen.Width = 1;
            if (maxTime > 0) // draw interpolated graph
            {
                for (j = 0; j < maxPoints - 1; j++)
                {
                    if ((xp[j] >= minX) && (xp[j] <= maxX))
                    {
                        p1.X = Convert.ToInt32(xp[j] * pGraph.Width / (double)maxTime);
                        p2.X = Convert.ToInt32(xp[j + 1] * pGraph.Width / (double)maxTime);
                        if (drawCeroline)
                        {
                            p1.Y = (pGraph.Height / 2) - Convert.ToInt32(Math.Round(yp[j] * scalefactor));
                            p2.Y = (pGraph.Height / 2) - Convert.ToInt32(Math.Round(yp[j + 1] * scalefactor));
                        }
                        else
                        {
                            p1.Y = (pGraph.Height) - Convert.ToInt32(Math.Round(yp[j] * scalefactor));
                            p2.Y = (pGraph.Height) - Convert.ToInt32(Math.Round(yp[j + 1] * scalefactor));
                        }
                        if (p2.X > p1.X)
                            g.DrawLine(bluePen, p1, p2);
                    }
                }
                for (j = 0; j < order; j++) // mark reference points
                {
                    p1.X = Convert.ToInt32(x_k[j] * pGraph.Width / (double)maxTime);
                    if (drawCeroline)
                        p1.Y = (pGraph.Height / 2) - Convert.ToInt32(Math.Round(y_k[j] * scalefactor));
                    else
                        p1.Y = (pGraph.Height) - Convert.ToInt32(Math.Round(y_k[j] * scalefactor));
                    g.DrawEllipse(redPen, (int)(p1.X - 1.0), (int)(p1.Y - 1.0), 2, 2);
                }

            }
            g.Dispose();
        }


        private void button1_Click(object sender, EventArgs e)
        {
            int i;
            deltaX = (b - a) / (double)order;
 
            // initialize supporting points
            i = 0;
            for (double x = a; x <= b; x += deltaX)
            {
                x_k[i] = x;
                y_k[i] = f_x(x);
                i++;
            }
            integral = 0;
            for (i = 0; i <= order; i++)
            {
                integral = integral + y_k[i] * CalcWeights(i);
            }

            tbIntegral.Text = integral.ToString();

            Interpolate(datapoints, b);
            DrawGraph(datapoints, a, b, true);
        }

        private void pGraph_Paint(object sender, PaintEventArgs e)
        {
            DrawGraph(datapoints, a, b, false);
        }  
    }
}
