﻿
using System;
using System.Drawing;
using System.Windows.Forms;
using System.Diagnostics;

/*  Interpolation by natural cubic splines */
/*  www.mosismaht.com                      */


namespace WindowsFormsApplication1
{

    public partial class MainWin : Form
    {
        public const int order = 6; // number of reference points (max 10)
        public const int datapoints = 1000; // number of datapoints for the interpolation
        public const double deltaX = 0.1; // sample time 

        public TextBox[,] tbm;
        public TextBox[] tbs;
        public TextBox[] tbx;
        public double[] yp = new double[datapoints];
        public double[] xp = new double[datapoints];
        public double[] y_k = new double[order];
        public double[] x_k = new double[order];
        public double[] a = new double[order];
        public double[] b = new double[order];
        public double[] c = new double[order];
        public double[] d = new double[order];
        public double[] h = new double[order - 1];
        public double[,] m = new double[order-2, order-2];
        public double[] y = new double[order-2];
        public double[] x = new double[order-2];

       
        public MainWin()
        {
            int i;
            tbs = new TextBox[10];
            tbx = new TextBox[10];

            InitializeComponent();

            tbs[0] = textBoxY1; tbs[1] = textBoxY2; tbs[2] = textBoxY3; tbs[3] = textBoxY4; tbs[4] = textBoxY5;
            tbs[5] = textBoxY6; tbs[6] = textBoxY7; tbs[7] = textBoxY8; tbs[8] = textBoxY9; tbs[9] = textBoxY10;

            tbx[0] = textBoxX1; tbx[1] = textBoxX2; tbx[2] = textBoxX3; tbx[3] = textBoxX4; tbx[4] = textBoxX5;
            tbx[5] = textBoxX6; tbx[6] = textBoxX7; tbx[7] = textBoxX8; tbx[8] = textBoxX9; tbx[9] = textBoxX10;
 
            // init some sample y values
            y_k[0] = 0.5;
            y_k[1] = 11.7;
            y_k[2] = 14.8;
            y_k[3] = 4.0;
            y_k[4] = 2.2;
            y_k[5] = 0.2;

            // timestamp to y value
            for (i = 1; i <= order; i++)
            {
                x_k[i - 1] = deltaX * (double)i;
            }

            if (order <= 10)
            {
                for (i = 0; i < order; i++)
                {
                    tbs[i].Text = Convert.ToString(y_k[i]);
                    tbx[i].Text = Convert.ToString(x_k[i]);

                }
                for (i = order; i < 10; i++)
                {
                    tbs[i].Visible = false;
                    tbx[i].Visible = false;
                }
            }
        }

        public bool CalcParameters()
        {
            int i, k;
            for (i = 0; i < order; i++)  // calculate a parameters
            {
                a[i] = y_k[i];
            }
            for (i = 0; i < order - 1; i++)
            {
                h[i] = x_k[i + 1] - x_k[i];
            }

            // empty matrix for c parameters
            for (i = 0; i < order - 2; i++)
            {
                for (k = 0; k < order - 2; k++)
                {
                    m[i, k] = 0.0;
                    y[i] = 0.0;
                    x[i] = 0.0;
                }
            }

            // fill matrix for c parameters
            for (i = 0; i < order - 2; i++)
            {
                if (i == 0)
                {
                    m[i, 0] = 2.0 * (h[0] + h[1]);
                    m[i, 1] = h[1];
                }
                else
                {
                    m[i, i - 1] = h[i];
                    m[i, i] = 2.0 * (h[i] + h[i + 1]);
                    if (i < order - 3)
                        m[i, i + 1] = h[i + 1];
                }
                if ((h[i] != 0.0) && (h[i + 1] != 0.0))
                    y[i] = ((a[i + 2] - a[i + 1]) / h[i + 1] - (a[i + 1] - a[i]) / h[i]) * 3.0;
                else
                    y[i] = 0.0;
            }

            TGauss gauss = new TGauss(order-2, m , y);

            if (gauss.Eliminate())
            {
                x = gauss.Solve();
                c[0] = 0.0;  // boundry condition for natural spline 
                for (i = 1; i < order - 1; i++)
                {
                    c[i] = x[i - 1];
                }
                for (i = 0; i < order - 1; i++) // calculate b and d parameters
                {
                    if (h[i] != 0.0)
                    {
                        d[i] = 1.0 / 3.0 / h[i] * (c[i + 1] - c[i]);
                        b[i] = 1.0 / h[i] * (a[i + 1] - a[i]) - h[i] / 3.0 * (c[i + 1] + 2 * c[i]);
                    }
                }
                return true;
            }
            else
                return false;
        }

        public void Interpolate(int order, int resolution)
        {
            int i, k;
            double timestamp;
            for (i = 0; i < order - 1; i++)
            {
                for (k = 0; k < resolution; k++)
                {
                    timestamp = (double)(k) / (double)(resolution) * h[i];
                    yp[i * resolution + k] = a[i] + b[i] * (timestamp) + c[i] * Math.Pow(timestamp, 2) + d[i] * Math.Pow(timestamp, 3);
                    xp[i * resolution + k] = timestamp + x_k[i];
                }
            }
        }

        public void DrawGraph(int maxPoints, bool doClear)
        {
            Point p1, p2;
            int j;
            double maxTime = -1.0;
            double maxVal  = -1.0;
            double minVal  = 1.0;
            double scalefactor;
            bool drawCeroline;
            for (j = 0; j < maxPoints - 1; j++)
            {
                if (maxTime < xp[j])
                    maxTime = xp[j];
                if (maxVal < yp[j])
                    maxVal = yp[j];
                if (minVal > yp[j])
                    minVal = yp[j];
            }
            maxTime = maxTime * 1.05;
            if (minVal < 0)
            {
                if (maxVal > 0)
                {
                    if (maxVal > Math.Abs(minVal))
                        scalefactor = pGraph.Height / maxVal / 2.2;
                    else
                        scalefactor = pGraph.Height / Math.Abs(minVal) / 2.2;
                }
                else
                    scalefactor = 5.0;
                drawCeroline = true;
            }
            else
            {
                scalefactor = pGraph.Height / (maxVal) / 1.1;
                drawCeroline = false;
            }
            p1 = new Point();
            p2 = new Point();
            Graphics g = pGraph.CreateGraphics();
            if (doClear)
                g.Clear(Color.White);
            Pen bluePen = new Pen(Color.Blue, 2);
            Pen redPen = new Pen(Color.Red, 2);
            Pen blackPen = new Pen(Color.Black, 2);
            if (drawCeroline)
            {
                p1.X = 0;
                p1.Y = (pGraph.Height / 2);
                p2.X = pGraph.Width;
                p2.Y = (pGraph.Height / 2);
                g.DrawLine(blackPen, p1, p2);
            }
            bluePen.Width = 1;
            if (maxTime > 0) // draw interpolated graph
            {
                for (j = 0; j < maxPoints - 1; j++)
                {
                    p1.X = Convert.ToInt32(xp[j] * pGraph.Width / (double)maxTime);
                    p2.X = Convert.ToInt32(xp[j + 1] * pGraph.Width / (double)maxTime);
                    if (drawCeroline)
                    {
                        p1.Y = (pGraph.Height / 2) - Convert.ToInt32(Math.Round(yp[j] * scalefactor));
                        p2.Y = (pGraph.Height / 2) - Convert.ToInt32(Math.Round(yp[j + 1] * scalefactor));
                    }
                    else
                    {
                        p1.Y = (pGraph.Height) - Convert.ToInt32(Math.Round(yp[j] * scalefactor));
                        p2.Y = (pGraph.Height) - Convert.ToInt32(Math.Round(yp[j + 1] * scalefactor));
                    }
                    if (p2.X > p1.X)
                        g.DrawLine(bluePen, p1, p2);
                }
                for (j = 0; j < order; j++) // mark reference points
                {
                    p1.X = Convert.ToInt32(x_k[j] * pGraph.Width / (double)maxTime);
                    if (drawCeroline)
                        p1.Y = (pGraph.Height / 2) - Convert.ToInt32(Math.Round(y_k[j] * scalefactor));
                    else
                        p1.Y = (pGraph.Height) - Convert.ToInt32(Math.Round(y_k[j] * scalefactor));
                    g.DrawEllipse(redPen, (int)(p1.X - 1.0), (int)(p1.Y - 1.0), 2, 2);
                }
            }
            g.Dispose();  
        }

        private void button1_MouseClick(object sender, MouseEventArgs e)
        {
            int i;
            bool isError = false;
            Stopwatch timer = new Stopwatch();
            if (order <= 10)
            {
                for (i = 0; i < order; i++)
                {
                    if (tbs[i].Text != "")
                    {
                        y_k[i] = Convert.ToDouble(tbs[i].Text);
                        x_k[i] = Convert.ToDouble(tbx[i].Text);
                    }
                    else
                    {
                        y_k[i] = 0.0;
                        x_k[i] = 0.0;
                    }
                    if (i > 0)
                    {
                        if (x_k[i] < x_k[i - 1])
                            isError = true;
                    }
                }
                if (isError)
                    MessageBox.Show("Time stamps must increase", "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
            }
            timer.Start();
            if (CalcParameters())
            {
                Interpolate(order, datapoints / order);
                DrawGraph(datapoints, true);
            }
            else
                MessageBox.Show("Matrix calculattion", "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);

            timer.Stop();
            tbTime.Text = timer.ElapsedMilliseconds.ToString();
        }


        private void pGraph_Paint(object sender, PaintEventArgs e)
        {
            DrawGraph(datapoints, false);
        }

    }
}
